#include <stdio.h>
#include <stdlib.h>
#include <mpi.h>

#define HISTOGRAM_BIN_COUNT 4
#define N 32
#define MASTER 0
#define TAG_GENERAL 1

int main(int argc, char **argv) {
    int rank, nprocs;
    MPI_Init(&argc, &argv);
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);
    MPI_Comm_size(MPI_COMM_WORLD, &nprocs);

    // Root allocates input and prints it
    char *Color = NULL;
    int  *Histogram = NULL;

    if (rank == MASTER) {
        Color = (char*)malloc(N * sizeof(char));
        Histogram = (int*)calloc(HISTOGRAM_BIN_COUNT, sizeof(int));
        srand(2017);
        for (int i = 0; i < N; ++i) Color[i] = (char)(rand() % HISTOGRAM_BIN_COUNT);

        for (int i = 0; i < N; ++i) printf("%3d ", Color[i]);
        printf("\n"); fflush(stdout);
    }

    // Build Scatterv metadata
    int *sendcounts = (int*)malloc(nprocs * sizeof(int));
    int *displs     = (int*)malloc(nprocs * sizeof(int));

    int base = N / nprocs;
    int rem  = N % nprocs;
    int off  = 0;
    for (int p = 0; p < nprocs; ++p) {
        sendcounts[p] = base + (p < rem ? 1 : 0);   // fair split
        displs[p]     = off;
        off          += sendcounts[p];
    }

    // Allocate receive slice per rank
    int rcount = sendcounts[rank];
    char *slice = (rcount > 0) ? (char*)malloc(rcount * sizeof(char)) : NULL;

    // Variable-size scatter
    MPI_Scatterv(Color, sendcounts, displs, MPI_CHAR,
                 slice, rcount, MPI_CHAR, MASTER, MPI_COMM_WORLD);

    // Local histogram
    int local_hist[HISTOGRAM_BIN_COUNT] = {0};
    for (int i = 0; i < rcount; ++i) {
        unsigned int bin = (unsigned char)slice[i];
        if (bin < HISTOGRAM_BIN_COUNT) local_hist[bin]++;
    }

    // Global reduction (single call)
    int global_hist[HISTOGRAM_BIN_COUNT] = {0};
    MPI_Reduce(local_hist, global_hist, HISTOGRAM_BIN_COUNT,
               MPI_INT, MPI_SUM, MASTER, MPI_COMM_WORLD);

    // Print on root
    if (rank == MASTER) {
        printf("---\n");
        for (int c = 0; c < HISTOGRAM_BIN_COUNT; ++c)
            printf("c=%3d  d=%d\n", c, global_hist[c]);
        fflush(stdout);
    }

    // Cleanup
    if (slice) free(slice);
    free(sendcounts);
    free(displs);
    if (rank == MASTER) { free(Color); free(Histogram); }

    MPI_Finalize();
    return 0;
}
